﻿using Harmony;
using RimWorld;
using RimWorld.Planet;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using UnityEngine;
using Verse;

namespace RFC_Code {
	
	[StaticConstructorOnStartup]
	internal static class RFC_Initializer {
		static RFC_Initializer() {
			LongEventHandler.QueueLongEvent(new Action(RFC_Initializer.Setup), "LibraryStartup", false, null);
		}
		public static void Setup() {
			SetIncidents.SetIncidentLevels();
		}
	}

	public class Controller : Mod {
		public static Dictionary<Faction, int> factionCenters = new Dictionary<Faction, int>();
		public static Settings Settings;
		public static double minFactionSeparation = 0;
		public static double maxFactionSprawl = 0;
		public override string SettingsCategory() { return "RFC.FactionControl".Translate(); }
		public override void DoSettingsWindowContents(Rect canvas) { Settings.DoWindowContents(canvas); }
		public Controller(ModContentPack content) : base(content) {
			HarmonyInstance harmony = HarmonyInstance.Create("net.rainbeau.rimworld.mod.factioncontrol");
			harmony.PatchAll( Assembly.GetExecutingAssembly() );
			Settings = GetSettings<Settings>();
		}
	}

	public class Settings : ModSettings {
		public bool factionNations = false;
		public float factionDensity = 2.5f;
		public float factionCount = 5.5f;
		public float outlanderMin = 1.5f;
		public float tribalMin = 1.5f;
		public float customFriendMin = 1.5f;
		public float pirateMin = 1.5f;
		public float customEnemyMin = 1.5f;
		public bool allowMechanoids = true;
		public void DoWindowContents(Rect canvas) {
			Listing_Standard list = new Listing_Standard();
			list.ColumnWidth = canvas.width;
			list.Begin(canvas);
			list.Gap();
			list.CheckboxLabeled( "RFC.FactionNations".Translate(), ref factionNations, "RFC.FactionNationsTip".Translate() );
			list.Gap();
			if (factionDensity < 1) {
				list.Label("RFC.factionDensity".Translate()+"  "+"RFC.factionDensityVeryLow".Translate());
			}
			else if (factionDensity < 2 ) {
				list.Label("RFC.factionDensity".Translate()+"  "+"RFC.factionDensityLow".Translate());
			}
			else if (factionDensity < 3) {
				list.Label("RFC.factionDensity".Translate()+"  "+"RFC.factionDensityNormal".Translate());
			}
			else if (factionDensity < 4) {
				list.Label("RFC.factionDensity".Translate()+"  "+"RFC.factionDensityHigh".Translate());
			}
			else {
				list.Label("RFC.factionDensity".Translate()+"  "+"RFC.factionDensityVeryHigh".Translate());
			}
			factionDensity = list.Slider(factionDensity, 0, 4.99f);
			list.Gap();
			list.Label("RFC.factionCount".Translate()+"  "+(int)factionCount);
			factionCount = list.Slider(factionCount, 0, 30.99f);
			Text.Font = GameFont.Tiny;
			list.Label("RFC.factionNotes".Translate());
			Text.Font = GameFont.Small;
			list.Gap(24);
			list.Label("RFC.outlanderMin".Translate()+"  "+(int)outlanderMin);
			outlanderMin = list.Slider(outlanderMin, 0, 10.99f);
			list.Gap();
			list.Label("RFC.tribalMin".Translate()+"  "+(int)tribalMin);
			tribalMin = list.Slider(tribalMin, 0, 10.99f);
			list.Gap();
			list.Label("RFC.customFriendMin".Translate()+"  "+(int)customFriendMin);
			customFriendMin = list.Slider(customFriendMin, 0, 10.99f);
			list.Gap();
			list.Label("RFC.pirateMin".Translate()+"  "+(int)pirateMin);
			pirateMin = list.Slider(pirateMin, 0, 5.99f);
			list.Gap();
			list.Label("RFC.customEnemyMin".Translate()+"  "+(int)customEnemyMin);
			customEnemyMin = list.Slider(customEnemyMin, 0, 5.99f);
			list.Gap();
			list.CheckboxLabeled( "RFC.AllowMechanoids".Translate(), ref allowMechanoids, "RFC.AllowMechanoidsTip".Translate() );
			list.End();
		}
		public override void ExposeData() {
			base.ExposeData();
			Scribe_Values.Look(ref factionNations, "factionNations", false);
			Scribe_Values.Look(ref factionDensity, "factionDensity", 2.5f);
			Scribe_Values.Look(ref factionCount, "factionCount", 5.5f);
			Scribe_Values.Look(ref outlanderMin, "outlanderMin", 1.5f);
			Scribe_Values.Look(ref tribalMin, "tribalMin", 1.5f);
			Scribe_Values.Look(ref customFriendMin, "customFriendMin", 1.5f);
			Scribe_Values.Look(ref pirateMin, "pirateMin", 1.5f);
			Scribe_Values.Look(ref customEnemyMin, "customEnemyMin", 1.5f);
			Scribe_Values.Look(ref allowMechanoids, "allowMechanoids", true);
			SetIncidents.SetIncidentLevels();
		}
	}

	public static class SetIncidents {
		public static void SetIncidentLevels() {
			foreach (IncidentDef def in DefDatabase<IncidentDef>.AllDefsListForReading) {
				if (def.defName == "PoisonShipPartCrash" || def.defName == "PsychicEmanatorShipPartCrash") {
					if (Controller.Settings.allowMechanoids.Equals(true)) {
						def.baseChance = 2.0f;
					}
					else {
						def.baseChance = 0.0f;
					}
				}
			}
		}
	}
		
	[HarmonyPatch(typeof(IncidentWorker_RaidEnemy), "FactionCanBeGroupSource", null)]
	public static class IncidentWorker_RaidEnemy_FactionCanBeGroupSource {
		public static bool Prefix(Faction f, ref bool __result) {
			if (f == Faction.OfMechanoids) {
				int hostileCount = 0;
				List<Faction> allFactionsListForReading = Find.FactionManager.AllFactionsListForReading;
				for (int i = 0; i < allFactionsListForReading.Count; i++) {
					if ((allFactionsListForReading[i] != Faction.OfMechanoids) && !allFactionsListForReading[i].def.hidden && allFactionsListForReading[i].HostileTo(Faction.OfPlayer)) {
						hostileCount++;
					}
				}
				if (hostileCount < 1) {
					__result = true;
					return false;
				}
				if (Controller.Settings.allowMechanoids.Equals(false) && hostileCount > 0) {
					__result = false;
					return false;
				}
			}
			return true;
		}
	}

	[HarmonyPatch(typeof(FactionGenerator), "GenerateFactionsIntoWorld", null)]
	public static class FactionGenerator_GenerateFactionsIntoWorld {
		public static bool Prefix() {
			int num = 0;
			int customCount = 0;
			Controller.factionCenters.Clear();
			Controller.minFactionSeparation = Math.Sqrt(Find.WorldGrid.TilesCount)/(Math.Sqrt(Controller.Settings.factionCount)*2);
			Controller.maxFactionSprawl = Math.Sqrt(Find.WorldGrid.TilesCount)/(Math.Sqrt(Controller.Settings.factionCount)*3);
			foreach (FactionDef allDef in DefDatabase<FactionDef>.AllDefs) {
				if (allDef.defName == "Outlander") {
					allDef.requiredCountAtGameStart = (int)Controller.Settings.outlanderMin;
					if (allDef.requiredCountAtGameStart < 1) {
						allDef.maxCountAtGameStart = 0;
					}
					else {
						allDef.maxCountAtGameStart = 100;
					}
				}
				else if (allDef.defName == "Tribe") {
					allDef.requiredCountAtGameStart = (int)Controller.Settings.tribalMin;
					if (allDef.requiredCountAtGameStart < 1) {
						allDef.maxCountAtGameStart = 0;
					}
					else {
						allDef.maxCountAtGameStart = 100;
					}
				}
				else if (allDef.defName == "Pirate") {
					allDef.requiredCountAtGameStart = (int)Controller.Settings.pirateMin;
					allDef.maxCountAtGameStart = allDef.requiredCountAtGameStart * 2;
				}
				else if (allDef.defName == "Spacer" || allDef.defName == "SpacerHostile") { }
				else if (allDef.defName == "Mechanoid" || allDef.defName == "Insect") { }
				else if (allDef.isPlayer) { }
				else {
					if (allDef.startingGoodwill.max < 0) {
						allDef.requiredCountAtGameStart = (int)Controller.Settings.customEnemyMin;
						allDef.maxCountAtGameStart = allDef.requiredCountAtGameStart * 2;
					}
					else {
						customCount++;
						allDef.requiredCountAtGameStart = (int)Controller.Settings.customFriendMin;
						if (allDef.requiredCountAtGameStart < 1) {
							allDef.maxCountAtGameStart = 0;
						}
						else {
							allDef.maxCountAtGameStart = 100;
						}
					}
				}
				for (int i = 0; i < allDef.requiredCountAtGameStart; i++) {
					Faction faction = FactionGenerator.NewGeneratedFaction(allDef);
					Find.FactionManager.Add(faction);
					if (!allDef.hidden) {
						num++;
					}
				}
			}
			if ((FactionDef.Named("Outlander").maxCountAtGameStart < 1 && FactionDef.Named("Tribe").maxCountAtGameStart < 1)
			  && (customCount < 1 || (int)Controller.Settings.customFriendMin < 1)) {
				foreach (FactionDef allDef in DefDatabase<FactionDef>.AllDefs) {
					if (allDef.startingGoodwill.max < 0) {
						allDef.maxCountAtGameStart = 100;
					}
				}
			}
			while (num < (int)Controller.Settings.factionCount) {
				FactionDef factionDef = (
				  from fa in DefDatabase<FactionDef>.AllDefs
				  where (!fa.canMakeRandomly ? false : Find.FactionManager.AllFactions.Count<Faction>((Faction f) => f.def == fa) < fa.maxCountAtGameStart)
				  select fa).RandomElement<FactionDef>();
				Faction faction1 = FactionGenerator.NewGeneratedFaction(factionDef);
				
				Find.World.factionManager.Add(faction1);
				num++;
			}
			float tilesCount = (float)Find.WorldGrid.TilesCount / 100000f;
			float minBP100K = (int)Controller.Settings.factionCount * 15f;
			float maxBP100K = (int)Controller.Settings.factionCount * 17f;
			if (Controller.Settings.factionDensity < 1) {
				minBP100K = minBP100K * 0.25f;
				maxBP100K = maxBP100K * 0.25f;
			}
			else if (Controller.Settings.factionDensity < 2) {
				minBP100K = minBP100K * 0.5f;
				maxBP100K = maxBP100K * 0.5f;
			}
			else if (Controller.Settings.factionDensity < 3) { }
			else if (Controller.Settings.factionDensity < 4) {
				minBP100K = minBP100K * 2;
				maxBP100K = maxBP100K * 2;
			}
			else {
				minBP100K = minBP100K * 4;
				maxBP100K = maxBP100K * 4;
			}
			FloatRange factionBasesPer100kTiles = new FloatRange(minBP100K, maxBP100K);
			int count = GenMath.RoundRandom(tilesCount * factionBasesPer100kTiles.RandomInRange);
			count -= Find.WorldObjects.FactionBases.Count;
			for (int j = 0; j < count; j++) {
				Faction faction2 = (
				  from x in Find.World.factionManager.AllFactionsListForReading
				  where (x.def.isPlayer ? false : !x.def.hidden)
				  select x).RandomElementByWeight<Faction>((Faction x) => x.def.baseSelectionWeight);
				FactionBase factionBase = (FactionBase)WorldObjectMaker.MakeWorldObject(WorldObjectDefOf.FactionBase);
				factionBase.SetFaction(faction2);
				factionBase.Tile = TileFinder.RandomFactionBaseTileFor(faction2, false, null);
				factionBase.Name = FactionBaseNameGenerator.GenerateFactionBaseName(factionBase);
				Find.WorldObjects.Add(factionBase);
			}
			return false;
		}
	}

	[HarmonyPatch(typeof(TileFinder), "RandomFactionBaseTileFor", null)]
	public static class TileFinder_RandomFactionBaseTileFor {
		public static bool Prefix(Faction faction, ref int __result, bool mustBeAutoChoosable = false, Predicate<int> extraValidator = null) {
			if (Controller.Settings.factionNations.Equals(false)) {
				return true;
			}
			int num;
			for (int i = 0; i < 2500; i++) {
				if ((
				from _ in Enumerable.Range(0, 100)
				select Rand.Range(0, Find.WorldGrid.TilesCount)).TryRandomElementByWeight<int>((int x) => {
					Tile item = Find.WorldGrid[x];
					if (!item.biome.canBuildBase || !item.biome.implemented || item.hilliness == Hilliness.Impassable) {
						return 0f;
					}
					if (mustBeAutoChoosable && !item.biome.canAutoChoose) {
						return 0f;
					}
					if (extraValidator != null && !extraValidator(x)) {
						return 0f;
					}
					return item.biome.factionBaseSelectionWeight;
				}, out num)) {
					if (TileFinder.IsValidTileForNewSettlement(num, null)) {
						if (faction == null || faction.def.hidden.Equals(true) || faction.def.isPlayer.Equals(true)) {
							__result = num;
							return false;
						}
						else if (Controller.factionCenters.ContainsKey(faction)) { 
							float test = Find.WorldGrid.ApproxDistanceInTiles(Controller.factionCenters[faction],num);
							if (test < Controller.maxFactionSprawl) { 
								__result = num;
								return false;
							}
						}
						else {
							bool locationOK = true;
							foreach (KeyValuePair<Faction,int> factionCenter in Controller.factionCenters) {
								float test = Find.WorldGrid.ApproxDistanceInTiles(factionCenter.Value,num);
								if (test < Controller.minFactionSeparation) {
									locationOK = false;
								}
							}
							if (locationOK.Equals(true)) {
								__result = num;
								Controller.factionCenters.Add(faction, num);
								return false;
							}
						}
					}
				}
			}
			Log.Warning(string.Concat("Failed to find faction base tile for ", faction, ". Relocating."));
			if (Controller.factionCenters.ContainsKey(faction)) {
				Controller.factionCenters.Remove(faction);
			}
			__result = 0;
			return false;
		}
	}

	[HarmonyPatch(typeof(FactionGenerator), "EnsureRequiredEnemies", null)]
	public static class FactionGenerator_EnsureRequiredEnemies {
		public static bool Prefix(Faction player) {
			foreach (FactionDef allDef in DefDatabase<FactionDef>.AllDefs) {
				if (!allDef.mustStartOneEnemy || !Find.World.factionManager.AllFactions.Any<Faction>((Faction f) => f.def == allDef) || Find.World.factionManager.AllFactions.Any<Faction>((Faction f) => (f.def != allDef ? false : f.HostileTo(player)))) {
					continue;
				}
				Faction faction = (
				from f in Find.World.factionManager.AllFactions
				where f.def == allDef
				select f).RandomElement<Faction>();
				float single = -(faction.GoodwillWith(player) + 100f) * Rand.Range(0.51f, 0.99f);
				faction.AffectGoodwillWith(player, single);
			}
			int hostileCount = 0;
			int friendlyCount = 0;
			List<Faction> allFactionsListForReading = Find.FactionManager.AllFactionsListForReading;
			for (int i = 0; i < allFactionsListForReading.Count; i++) {
				if (allFactionsListForReading[i].def.hidden || allFactionsListForReading[i].def.isPlayer) { continue; }
				if (allFactionsListForReading[i].GoodwillWith(player) > -80f) { allFactionsListForReading[i].SetHostileTo(player, false); }
				if (allFactionsListForReading[i].HostileTo(player)) {
					hostileCount++;
				}
				else {
					friendlyCount++;
				}
			}
			int hostileTarget = (int)(Find.World.factionManager.AllFactions.Count()/3) + Rand.Range(-1,2);
			if ((hostileTarget-hostileCount) > friendlyCount) { hostileTarget = friendlyCount + hostileCount; }
			for (int i = hostileCount; i < hostileTarget; i++) {
				Faction faction = (
				from f in Find.World.factionManager.AllFactions
				where (f.HostileTo(player) ? false : f.def.isPlayer ? false : !f.def.hidden)
				select f).RandomElement<Faction>();
				float single = -(faction.GoodwillWith(player) + 100f) * Rand.Range(0.51f, 0.99f);
				faction.AffectGoodwillWith(player, single);
			}
			return false;
		}
	}
        
}
